/*
 * Decompiled with CFR 0.152.
 */
package com.ibm.nmon.parser;

import com.ibm.nmon.data.BasicDataSet;
import com.ibm.nmon.data.DataRecord;
import com.ibm.nmon.data.DataType;
import com.ibm.nmon.data.SubDataType;
import com.ibm.nmon.util.DataHelper;
import com.ibm.nmon.util.TimeHelper;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class IOStatParser {
    private static final Logger LOGGER = LoggerFactory.getLogger(IOStatParser.class);
    private static final SimpleDateFormat TIMESTAMP_FORMAT_US = new SimpleDateFormat("MM/dd/yy HH:mm:ss");
    private static final SimpleDateFormat TIMESTAMP_FORMAT_OLD = new SimpleDateFormat("'Time: 'hh:mm:ss a");
    private static final SimpleDateFormat TIMESTAMP_FORMAT_AIX = new SimpleDateFormat("HH:mm:ss");
    private static final SimpleDateFormat DATE_FORMAT_US = new SimpleDateFormat("MM/dd/yyyy");
    private static final Matcher ISO_PATTERN = Pattern.compile("(Time: )?\\d{4}\\-\\d{2}\\-\\d{2}T\\d{2}:\\d{2}:\\d{2}([\\-+](\\d{4}?|\\d{2}:\\d{2}|\\d{2})|Z)").matcher("");
    private static final Matcher INFO = Pattern.compile("(.+)\\s(.+)\\s\\((.+)\\)\\s+(\\d{2,4}[\\/-]\\d{2}[\\/-]\\d{2,4})(\\s+_(.+)_)?(\\s+\\((.+)\\sCPU\\))?").matcher("");
    private static final Pattern DATA_SPLITTER = Pattern.compile(":?\\s+");
    public static final String DEFAULT_HOSTNAME = "iostat";
    private LineNumberReader in = null;
    private SimpleDateFormat format = null;
    private boolean isAIX = false;
    private long dateOffset = 0L;
    private BasicDataSet data = null;
    private DataRecord currentRecord = null;
    private Map<String, List<String>> typesByHeader = new HashMap<String, List<String>>();

    public BasicDataSet parse(File file, TimeZone timeZone) throws IOException, ParseException {
        return this.parse(file.getAbsolutePath(), timeZone);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BasicDataSet parse(String filename, TimeZone timeZone) throws IOException, ParseException {
        long start = System.nanoTime();
        this.data = new BasicDataSet(filename);
        this.data.setHostname(DEFAULT_HOSTNAME);
        String line = null;
        try {
            this.in = new LineNumberReader(new FileReader(filename));
            this.parseHeader(timeZone);
            this.determineTimestampFormat(timeZone);
            line = this.in.readLine();
            if (this.isAIX) {
                if (line.startsWith("tty")) {
                    this.parseAIXTTYAndCPUHeader(line);
                    line = this.in.readLine();
                    if (line.startsWith("System")) {
                        this.data.setHostname(line.substring("System: ".length()));
                        line = this.in.readLine();
                    }
                }
            } else {
                if (line.startsWith("avg-cpu:")) {
                    this.parseLinuxCPUHeader(line);
                }
                line = this.in.readLine();
            }
            line = this.parseDataTypes(line);
            while (line != null) {
                String[] temp;
                String typeName;
                if (this.isAIX) {
                    if (line.startsWith("System")) {
                        line = this.in.readLine();
                        if (!"".equals(line)) continue;
                        line = this.in.readLine();
                        continue;
                    }
                } else if (line.startsWith("Time:") || ISO_PATTERN.reset(line).matches()) {
                    this.createCurrentRecord(line);
                    line = this.in.readLine();
                    continue;
                }
                if ("FS".equals(typeName = (temp = DATA_SPLITTER.split(line))[0])) {
                    typeName = temp[0] + ' ' + temp[1];
                }
                if ("tty".equals(typeName)) {
                    this.parseAIXTTYAndCPU();
                } else if ("avg-cpu".equals(typeName)) {
                    this.parseLinuxCPU();
                } else if ("".equals(typeName)) {
                    if (line.equals("")) {
                        line = this.in.readLine();
                        continue;
                    }
                    this.parseAIXSummaryData();
                } else {
                    this.parseData(typeName);
                }
                line = this.in.readLine();
            }
            if (this.currentRecord != null) {
                this.data.addRecord(this.currentRecord);
            }
            BasicDataSet basicDataSet = this.data;
            return basicDataSet;
        }
        finally {
            if (this.in != null) {
                try {
                    this.in.close();
                }
                catch (Exception e) {}
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Parse complete for {} in {}ms", (Object)this.data.getSourceFile(), (Object)((double)(System.nanoTime() - start) / 1000000.0));
                }
                this.in = null;
                this.format = null;
                this.isAIX = false;
                this.dateOffset = 0L;
                this.data = null;
                this.currentRecord = null;
                this.typesByHeader.clear();
            }
        }
    }

    private void parseHeader(TimeZone timeZone) throws IOException, ParseException {
        String line = this.in.readLine();
        while ("".equals(line)) {
            line = this.in.readLine();
        }
        if (line.startsWith("System configuration: ")) {
            this.isAIX = true;
            this.data.setMetadata("OS", "AIX");
            line = line.substring("System configuration: ".length());
            String[] config = line.split("[ =]");
            for (int i = 0; i < config.length; ++i) {
                this.data.setMetadata(DataHelper.newString(config[i]), DataHelper.newString(config[++i]));
            }
            this.dateOffset = IOStatParser.getDefaultDate();
        } else {
            Matcher matcher = INFO.reset(line);
            if (matcher.matches()) {
                this.data.setHostname(DataHelper.newString(matcher.group(3)));
                this.data.setMetadata("OS", DataHelper.newString(matcher.group(1) + ' ' + matcher.group(2)));
                String date = matcher.group(4);
                String arch = matcher.group(6);
                String cpuCount = matcher.group(8);
                SimpleDateFormat dateFormat = TimeHelper.DATE_FORMAT_ISO;
                if (date.indexOf(47) != -1) {
                    if (date.length() == DATE_FORMAT_US.toPattern().length() - 2) {
                        date = date.substring(0, 6) + "20" + date.substring(6);
                    }
                    dateFormat = DATE_FORMAT_US;
                    this.dateOffset = TimeHelper.dayFromDatetime(dateFormat.parse(date).getTime());
                }
                dateFormat.setTimeZone(timeZone);
                if (arch != null) {
                    this.data.setMetadata("ARCH", DataHelper.newString(arch));
                }
                if (cpuCount != null) {
                    this.data.setMetadata("CPU_COUNT", DataHelper.newString(cpuCount));
                }
            } else {
                throw new IOException("file does not start with a recognized Linux iostat header");
            }
        }
        this.dateOffset += (long)timeZone.getOffset(this.dateOffset);
        this.in.readLine();
    }

    private void determineTimestampFormat(TimeZone timeZone) throws IOException {
        if (this.isAIX) {
            this.format = TIMESTAMP_FORMAT_AIX;
            this.format.setTimeZone(timeZone);
        } else {
            String line = this.in.readLine();
            if (line.startsWith("Time: ")) {
                if (ISO_PATTERN.reset(line).matches()) {
                    this.format = new SimpleDateFormat("'Time: '" + TimeHelper.TIMESTAMP_FORMAT_ISO.toPattern());
                } else {
                    this.format = TIMESTAMP_FORMAT_OLD;
                    this.format.setTimeZone(timeZone);
                }
            } else {
                try {
                    TimeHelper.TIMESTAMP_FORMAT_ISO.parse(line);
                    this.format = TimeHelper.TIMESTAMP_FORMAT_ISO;
                }
                catch (ParseException pe) {
                    try {
                        TIMESTAMP_FORMAT_US.parse(line);
                        this.format = TIMESTAMP_FORMAT_US;
                        this.format.setTimeZone(timeZone);
                    }
                    catch (ParseException pe2) {
                        throw new IOException("unknown timestamp format");
                    }
                }
            }
        }
    }

    private void parseLinuxCPUHeader(String line) throws IOException {
        String[] temp = DATA_SPLITTER.split(line);
        String[] fields = new String[temp.length - 2];
        for (int i = 0; i < fields.length; ++i) {
            fields[i] = DataHelper.newString(temp[i + 1]);
        }
        DataType cpu = new DataType("IOStat CPU", "IOStat Average CPU", fields);
        this.data.addType(cpu);
        this.in.readLine();
        this.in.readLine();
    }

    private void parseAIXTTYAndCPUHeader(String line) throws IOException {
        String[] temp = DATA_SPLITTER.split(line);
        ArrayList<String> fields = new ArrayList<String>();
        for (int i = 1; i < temp.length; ++i) {
            if ("avg-cpu".equals(temp[i])) {
                DataType tty = new DataType("IOStat TTY", "IOStat terminal", fields.toArray(new String[0]));
                this.data.addType(tty);
                fields.clear();
                continue;
            }
            if ("time".equals(temp[i])) continue;
            if ("%".equals(temp[i])) {
                if ("idle".equals(temp[i + 1])) {
                    ++i;
                    continue;
                }
                fields.add(DataHelper.newString(temp[i++] + temp[i]));
                continue;
            }
            fields.add(DataHelper.newString(temp[i]));
        }
        DataType cpu = new DataType("IOStat CPU", "IOStat Average CPU", fields.toArray(new String[0]));
        this.data.addType(cpu);
        this.typesByHeader.put("tty", Arrays.asList("CPU", "TTY"));
        this.in.readLine();
        this.in.readLine();
    }

    private String parseDataTypes(String line) throws IOException {
        while (line != null) {
            int i;
            if (!this.isAIX && (line.startsWith("Time:") || ISO_PATTERN.reset(line).matches())) {
                return line;
            }
            String[] temp = DATA_SPLITTER.split(line);
            String type = temp[0];
            int offset = 1;
            List<String> subDataTypes = this.typesByHeader.get(type);
            if (subDataTypes != null) {
                if (this.isAIX) {
                    return line;
                }
                throw new IOException("duplicate header for " + type + " at line " + this.in.getLineNumber());
            }
            if (this.isAIX) {
                if ("".equals(type)) {
                    String[] fields = new String[temp.length - (offset + 1)];
                    for (i = offset; i < temp.length - 1; ++i) {
                        fields[i - 1] = DataHelper.newString(temp[i]);
                    }
                    line = this.in.readLine().trim();
                    while (!"".equals(line)) {
                        type = DataHelper.newString(DATA_SPLITTER.split(line)[0]);
                        this.data.addType(new DataType("IOStat " + type, type, fields));
                        line = this.in.readLine().trim();
                    }
                    line = this.in.readLine();
                    continue;
                }
                if ("Disks".equals(type) && "xfers".equals(temp[1])) {
                    throw new IOException("AIX extended disk statistics (-D) are not currently supported");
                }
                if ("System".equals(type)) {
                    this.in.readLine();
                    return this.in.readLine();
                }
                if ("FS".equals(type)) {
                    type = temp[0] + ' ' + temp[1];
                    ++offset;
                }
            }
            subDataTypes = new ArrayList<String>(temp.length - (offset + 1 + (this.isAIX ? 1 : 0)));
            int end = temp.length - (this.isAIX ? 1 : 0);
            for (i = offset; i < end; ++i) {
                if ("%".equals(temp[i])) {
                    subDataTypes.add(DataHelper.newString(temp[i] + temp[++i]));
                    continue;
                }
                subDataTypes.add(DataHelper.newString(temp[i]));
            }
            line = this.in.readLine();
            ArrayList<String> fields = new ArrayList<String>();
            while (!line.equals("")) {
                temp = DATA_SPLITTER.split(line);
                fields.add(DataHelper.newString(temp[0]));
                line = this.in.readLine();
            }
            if (fields.size() == 0) {
                LOGGER.warn("no fields defined for {}; data will be ignored", (Object)type);
            } else {
                for (String subType : subDataTypes) {
                    String[] fieldsArray = fields.toArray(new String[fields.size()]);
                    String name = type + ' ' + subType;
                    this.data.addType(new SubDataType("IOStat " + type, subType, name, false, fieldsArray));
                }
                this.typesByHeader.put(type, subDataTypes);
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("{} section contains {} DataTypes: {}", new Object[]{type, subDataTypes.size(), subDataTypes});
                    LOGGER.trace("{} section has {} {}: {}", new Object[]{type, fields.size(), fields.size() > 1 ? "ies" : "y", fields});
                }
            }
            line = this.in.readLine();
        }
        return line;
    }

    private void parseLinuxCPU() throws IOException {
        String[] temp = DATA_SPLITTER.split(this.in.readLine());
        double[] cpuData = new double[temp.length - 2];
        for (int i = 0; i < cpuData.length; ++i) {
            cpuData[i] = Double.parseDouble(temp[i + 1]);
        }
        this.currentRecord.addData(this.data.getType("IOStat CPU"), cpuData);
        this.in.readLine();
    }

    private void parseAIXTTYAndCPU() throws IOException, ParseException {
        String[] temp = DATA_SPLITTER.split(this.in.readLine());
        this.createCurrentRecord(temp[temp.length - 1]);
        int n = 1;
        DataType tty = this.data.getType("IOStat TTY");
        double[] ttyData = new double[tty.getFieldCount()];
        for (int i = 0; i < ttyData.length; ++i) {
            ttyData[i] = Double.parseDouble(temp[n++]);
        }
        this.currentRecord.addData(tty, ttyData);
        DataType cpu = this.data.getType("IOStat CPU");
        double[] cpuData = new double[cpu.getFieldCount()];
        int i = 0;
        while (i < cpuData.length) {
            cpuData[i++] = Double.parseDouble(temp[n++]);
        }
        this.currentRecord.addData(cpu, cpuData);
        this.in.readLine();
    }

    private void parseAIXSummaryData() throws IOException {
        String line = this.in.readLine();
        while (!"".equals(line)) {
            String[] temp = DATA_SPLITTER.split(line.trim());
            DataType type = this.data.getType("IOStat " + temp[0]);
            double[] data = new double[type.getFieldCount()];
            for (int i = 0; i < data.length; ++i) {
                data[i] = Double.parseDouble(temp[i + 1]);
            }
            this.currentRecord.addData(type, data);
            line = this.in.readLine();
        }
    }

    private void parseData(String type) throws IOException, ParseException {
        List<String> subTypes = this.typesByHeader.get(type);
        if (subTypes == null) {
            return;
        }
        HashMap<DataType, double[]> dataToAdd = new HashMap<DataType, double[]>();
        for (int i = 0; i < subTypes.size(); ++i) {
            String subType = subTypes.get(i);
            DataType dataType = this.data.getType(SubDataType.buildId("IOStat " + type, subType));
            dataToAdd.put(dataType, new double[dataType.getFieldCount()]);
        }
        int subTypeCount = subTypes.size();
        String line = this.in.readLine();
        while (line != null && !"".equals(line)) {
            String[] temp = DATA_SPLITTER.split(line);
            String field = temp[0];
            int dataLength = temp.length - (this.isAIX ? 2 : 1);
            if (this.isAIX && this.currentRecord == null) {
                this.createCurrentRecord(temp[temp.length - 1]);
            }
            if (dataLength > subTypeCount) {
                LOGGER.warn("'{}' at line {} has {} extra columns; they will be ignored", new Object[]{field, this.in.getLineNumber(), dataLength - subTypeCount});
                dataLength = subTypeCount;
            } else if (dataLength < subTypeCount) {
                LOGGER.warn("'{}' at line {} has too few columns; zero will be assumed for missing data", new Object[]{field, this.in.getLineNumber()});
            }
            for (int i = 0; i < subTypes.size(); ++i) {
                String subType = subTypes.get(i);
                DataType dataType = this.data.getType(SubDataType.buildId("IOStat " + type, subType));
                double[] subTypeData = (double[])dataToAdd.get(dataType);
                if (subTypeData == null) {
                    subTypeData = new double[dataType.getFieldCount()];
                    dataToAdd.put(dataType, subTypeData);
                }
                int subTypeIdx = dataType.getFieldIndex(field);
                String data = temp[i + 1];
                subTypeData[subTypeIdx] = "-".equals(data) ? Double.NaN : Double.parseDouble(data);
            }
            line = this.in.readLine();
        }
        for (DataType dataType : dataToAdd.keySet()) {
            double[] toAdd = (double[])dataToAdd.get(dataType);
            this.currentRecord.addData(dataType, toAdd);
        }
    }

    private void createCurrentRecord(String timeToParse) throws ParseException {
        if (this.currentRecord != null) {
            this.data.addRecord(this.currentRecord);
            this.currentRecord = null;
        }
        long time = this.format.parse(timeToParse).getTime() + this.dateOffset;
        this.currentRecord = new DataRecord(time, timeToParse);
    }

    public static long getDefaultDate() {
        return TimeHelper.today();
    }
}

