/*
 * Decompiled with CFR 0.152.
 */
package com.ibm.nmon.parser;

import com.ibm.nmon.data.DataRecord;
import com.ibm.nmon.data.DataType;
import com.ibm.nmon.data.NMONDataSet;
import com.ibm.nmon.data.Process;
import com.ibm.nmon.data.ProcessDataType;
import com.ibm.nmon.data.transform.AIXCPUTransform;
import com.ibm.nmon.data.transform.AIXLPARTransform;
import com.ibm.nmon.data.transform.AIXMemoryTransform;
import com.ibm.nmon.data.transform.CPUBusyTransform;
import com.ibm.nmon.data.transform.DataPostProcessor;
import com.ibm.nmon.data.transform.DataTransform;
import com.ibm.nmon.data.transform.DiskTotalTransform;
import com.ibm.nmon.data.transform.EthernetTotalPostProcessor;
import com.ibm.nmon.data.transform.LinuxMemoryTransform;
import com.ibm.nmon.data.transform.LinuxNetPacketTransform;
import com.ibm.nmon.data.transform.NetworkTotalPostProcessor;
import com.ibm.nmon.util.DataHelper;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class NMONParser {
    private static final Logger LOGGER = LoggerFactory.getLogger(NMONParser.class);
    private static final SimpleDateFormat NMON_FORMAT = new SimpleDateFormat("HH:mm:ss dd-MMM-yyyy", Locale.US);
    private static final Pattern DATA_SPLITTER = Pattern.compile(",");
    private LineNumberReader in = null;
    private DataRecord currentRecord = null;
    private NMONDataSet data = null;
    private String[] topFields = null;
    private int topCommandIndex = -1;
    private int fileCPUs = 1;
    private boolean seenFirstDataType = false;
    private boolean isAIX = false;
    private boolean scaleProcessesByCPU = true;
    private final Map<Integer, Process> processes = new HashMap<Integer, Process>();
    private final Map<String, StringBuilder> systemInfo = new HashMap<String, StringBuilder>();
    private final List<DataTransform> transforms = new ArrayList<DataTransform>();
    private final List<DataPostProcessor> processors = new ArrayList<DataPostProcessor>();
    private static final Set<String> IGNORED_TYPES = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("AVM-IN-MB", "NO-PBUF-COUNT", "NO-PSBUF-COUNT", "NO-JFS2-FSBUF-COUNT")));
    private static final Map<String, List<Integer>> TYPE_SKIP_INDEXES;

    public NMONParser() {
        this.processors.add(new NetworkTotalPostProcessor("NET"));
        this.processors.add(new NetworkTotalPostProcessor("SEA"));
        this.processors.add(new EthernetTotalPostProcessor("NET"));
        this.processors.add(new EthernetTotalPostProcessor("SEA"));
    }

    public NMONDataSet parse(File file, TimeZone timeZone, boolean scaleProcessesByCPU) throws IOException {
        return this.parse(file.getAbsolutePath(), timeZone, scaleProcessesByCPU);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NMONDataSet parse(String filename, TimeZone timeZone, boolean scaleProcessesByCPU) throws IOException {
        long start = System.nanoTime();
        this.scaleProcessesByCPU = scaleProcessesByCPU;
        this.in = new LineNumberReader(new FileReader(filename));
        try {
            this.data = new NMONDataSet(filename);
            NMON_FORMAT.setTimeZone(timeZone);
            this.data.setMetadata("parsed_gmt_offset", Double.toString((double)timeZone.getOffset(System.currentTimeMillis()) / 3600000.0));
            String line = this.parseHeaders();
            if (line == null || !line.startsWith("ZZZZ")) {
                throw new IOException("file '" + filename + "' does not appear to have any data records");
            }
            for (DataPostProcessor processor : this.processors) {
                processor.addDataTypes(this.data);
            }
            do {
                this.parseLine(line);
            } while ((line = this.in.readLine()) != null);
            for (String name : this.systemInfo.keySet()) {
                String value = this.systemInfo.get(name).toString();
                this.data.setSystemInfo(name, value);
            }
            if (this.currentRecord != null) {
                this.completeCurrentRecord();
            }
            DataHelper.aggregateProcessData(this.data, LOGGER);
            NMONDataSet nMONDataSet = this.data;
            return nMONDataSet;
        }
        finally {
            if (this.in != null) {
                try {
                    this.in.close();
                }
                catch (Exception e) {}
                this.in = null;
            }
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Parse complete for {} in {}ms", (Object)this.data.getSourceFile(), (Object)((double)(System.nanoTime() - start) / 1000000.0));
            }
            this.data = null;
            this.currentRecord = null;
            this.topFields = null;
            this.topCommandIndex = -1;
            this.fileCPUs = 1;
            this.seenFirstDataType = false;
            this.isAIX = false;
            this.processes.clear();
            this.systemInfo.clear();
            this.transforms.clear();
        }
    }

    private String parseHeaders() throws IOException {
        String line = null;
        while ((line = this.in.readLine()) != null) {
            DataType type;
            String[] values;
            if (line.startsWith("AAA")) {
                values = DATA_SPLITTER.split(line);
                if (values[1].startsWith("note") || values.length <= 2) continue;
                if ("OS".equals(values[1])) {
                    this.data.setMetadata("OS", DataHelper.newString(values[2] + ' ' + values[3]));
                    this.data.setMetadata("ARCH", DataHelper.newString(values[5]));
                    continue;
                }
                if ("MachineType".equals(values[1])) {
                    this.data.setMetadata("MachineType", DataHelper.newString(values[2] + ' ' + values[3]));
                    continue;
                }
                if ("LPARNumberName".equals(values[1])) {
                    if (values.length <= 3) continue;
                    this.data.setMetadata("LPARNumber", DataHelper.newString(values[2]));
                    this.data.setMetadata("LPARName", DataHelper.newString(values[3]));
                    continue;
                }
                if ("cpus".equals(values[1])) {
                    if (values.length == 4) {
                        this.data.setMetadata(DataHelper.newString(values[1]), DataHelper.newString(values[3]));
                        continue;
                    }
                    this.data.setMetadata(DataHelper.newString(values[1]), DataHelper.newString(values[2]));
                    continue;
                }
                this.data.setMetadata(DataHelper.newString(values[1]), DataHelper.newString(values[2]));
                continue;
            }
            if (line.startsWith("BBBP")) {
                this.parseBBBP(DATA_SPLITTER.split(line));
                continue;
            }
            if (line.startsWith("TOP")) {
                values = DATA_SPLITTER.split(line);
                if (!"+PID".equals(values[1])) continue;
                this.topFields = this.parseTopFields(values);
                continue;
            }
            if (line.startsWith("ZZZZ")) break;
            if (line.startsWith("BBB")) {
                this.parseSystemInfo(DATA_SPLITTER.split(line));
                continue;
            }
            if (line.startsWith("UARG") || line.isEmpty()) continue;
            if (!this.seenFirstDataType) {
                this.transforms.add(new CPUBusyTransform());
                this.transforms.add(new DiskTotalTransform());
                if (this.data.getMetadata("AIX") != null) {
                    this.isAIX = true;
                    this.transforms.add(new AIXMemoryTransform());
                    this.transforms.add(new AIXLPARTransform());
                    this.transforms.add(new AIXCPUTransform());
                } else {
                    this.transforms.add(new LinuxNetPacketTransform());
                    this.transforms.add(new LinuxMemoryTransform());
                }
                String temp = this.data.getMetadata("cpus");
                if (temp != null) {
                    try {
                        this.fileCPUs = Integer.parseInt(temp);
                    }
                    catch (NumberFormatException numberFormatException) {
                        // empty catch block
                    }
                }
                this.seenFirstDataType = true;
            }
            if ((type = this.buildDataType(DATA_SPLITTER.split(line))) == null) continue;
            this.data.addType(type);
        }
        return line;
    }

    private void parseLine(String line) {
        if (line.startsWith("ZZZZ")) {
            if (this.currentRecord != null) {
                this.completeCurrentRecord();
            }
            this.currentRecord = this.parseTimestamp(line);
        } else {
            if (line.startsWith("ERROR")) {
                return;
            }
            String[] values = DATA_SPLITTER.split(line);
            if (this.currentRecord == null) {
                if (IGNORED_TYPES.contains(values[0])) {
                    return;
                }
                throw new IllegalStateException("current record is null at line " + this.in.getLineNumber());
            }
            if (values.length < 2) {
                LOGGER.warn("skipping invalid data record '{}' starting at line {}", (Object)line, (Object)this.in.getLineNumber());
                return;
            }
            String timestamp = null;
            boolean isTop = "TOP".equals(values[0]);
            boolean isUarg = "UARG".equals(values[0]);
            timestamp = isTop ? values[2] : values[1];
            if (timestamp.startsWith("T")) {
                DataType type = this.data.getType(values[0]);
                if (timestamp.equals(this.currentRecord.getTimestamp())) {
                    if (isUarg) {
                        this.parseUARG(values);
                    } else if (isTop) {
                        this.parseTopData(values);
                    } else if (type == null) {
                        if ("VM".equals(values[0])) {
                            String[] newValues = new String[values.length - 1];
                            newValues[0] = values[0];
                            System.arraycopy(values, 2, newValues, 1, values.length - 2);
                            type = this.buildDataType(newValues);
                            this.data.addType(type);
                        } else {
                            LOGGER.warn("undefined data type {} at line {}", (Object)values[0], (Object)this.in.getLineNumber());
                        }
                    } else {
                        this.parseData(type, values);
                    }
                } else {
                    LOGGER.warn("misplaced record at line {}; expected timestamp {} but got {}", new Object[]{this.in.getLineNumber(), this.currentRecord.getTimestamp(), timestamp});
                }
            } else if (!isTop && !isUarg) {
                if ("BBBP".equals(values[0])) {
                    this.parseBBBP(values);
                } else if (!"AAA".equals(values[0])) {
                    DataType type;
                    if (values[0].startsWith("BBB")) {
                        this.parseSystemInfo(values);
                    } else if (this.data.getType(values[0]) == null && (type = this.buildDataType(values)) != null) {
                        if (type.getId().equals("NO-JFS2-FSBUF-COUNT")) {
                            this.completeCurrentRecord();
                        }
                        if (!IGNORED_TYPES.contains(type.getId())) {
                            this.data.addType(type);
                        }
                    }
                }
            }
        }
    }

    private DataRecord parseTimestamp(String line) {
        String[] values = DATA_SPLITTER.split(line);
        long time = 0L;
        if (values.length != 4) {
            LOGGER.warn("skipping invalid data record '{}' starting at line {}", (Object)line, (Object)this.in.getLineNumber());
            return null;
        }
        try {
            time = NMON_FORMAT.parse(values[2] + ' ' + values[3]).getTime();
            long previous = this.data.getEndTime();
            if (time < previous) {
                String temp = this.data.getMetadata("interval");
                if (temp == null) {
                    LOGGER.error("time {} is less than previous {} at line {}; no interval defined in AAA records", new Object[]{time, previous, this.in.getLineNumber()});
                    throw new IllegalArgumentException("time is less than previous in ZZZZ " + values[1]);
                }
                int interval = Integer.parseInt(temp);
                time = previous + (long)(interval * 1000);
                LOGGER.warn("time {} is less than previous {} at line {}, guessing at next time by using an interval of {}s", new Object[]{time, previous, this.in.getLineNumber(), interval});
            }
            DataRecord record = new DataRecord(time, DataHelper.newString(values[1]));
            return record;
        }
        catch (ParseException pe) {
            LOGGER.warn("could not parse time {}, {} at line {}", new Object[]{values[2], values[3], this.in.getLineNumber()});
            return null;
        }
    }

    private void parseBBBP(String[] values) {
        if (values.length == 4) {
            String command = DataHelper.newString(values[2]);
            StringBuilder builder = this.systemInfo.get(command);
            if (builder == null) {
                builder = new StringBuilder(256);
                this.systemInfo.put(command, builder);
            } else {
                builder.append('\n');
            }
            builder.append(values[3], 1, values[3].length() - 1);
        }
    }

    private void parseSystemInfo(String[] values) {
        StringBuilder builder = this.systemInfo.get(values[0]);
        if (builder == null) {
            builder = new StringBuilder(256);
            this.systemInfo.put(DataHelper.newString(values[0]), builder);
        } else {
            for (int i = 2; i < values.length - 1; ++i) {
                builder.append(values[i]);
                builder.append(',');
            }
            builder.append(values[values.length - 1]);
            builder.append('\n');
        }
    }

    private void parseData(DataType type, String[] values) {
        int i;
        List<Integer> toSkip = TYPE_SKIP_INDEXES.get(type.getId());
        if (toSkip == null) {
            toSkip = Collections.emptyList();
        }
        double[] recordData = new double[values.length - 2 - toSkip.size()];
        int n = 0;
        try {
            for (i = 2; i < values.length; ++i) {
                if (toSkip.contains(i)) continue;
                String data = values[i];
                recordData[n] = "".equals(data) || data.contains("nan") ? 0.0 : ("INF".equals(data) ? Double.POSITIVE_INFINITY : Double.parseDouble(data));
                ++n;
            }
        }
        catch (NumberFormatException nfe) {
            LOGGER.warn("{}: invalid numeric data '{}' at line {}, column {}", new Object[]{this.currentRecord.getTimestamp(), values[i], this.in.getLineNumber(), i + 1});
        }
        for (DataTransform transform : this.transforms) {
            if (!transform.isValidFor(type.getId(), null)) continue;
            try {
                recordData = transform.transform(type, recordData);
            }
            catch (Exception e) {
                LOGGER.warn(this.currentRecord.getTimestamp() + ": could not complete transform " + transform.getClass().getSimpleName() + " at line " + this.in.getLineNumber(), (Throwable)e);
            }
            break;
        }
        try {
            this.currentRecord.addData(type, recordData);
        }
        catch (IllegalArgumentException ile) {
            double[] newData = new double[type.getFieldCount()];
            System.arraycopy(recordData, 0, newData, 0, recordData.length);
            LOGGER.warn("{}: DataType {} defines {} fields but there are only {} values; missing values set to 0", new Object[]{this.currentRecord.getTimestamp(), type.getId(), type.getFieldCount(), recordData.length});
            recordData = newData;
            this.currentRecord.addData(type, recordData);
        }
    }

    private String[] parseTopFields(String[] values) {
        String[] topFields = new String[values.length - (this.isAIX ? 5 : 4) + 1];
        int valuesIdx = 3;
        int fieldsIdx = 0;
        while (valuesIdx < values.length) {
            if ("Command".equals(values[valuesIdx])) {
                this.topCommandIndex = valuesIdx++;
                continue;
            }
            if ("WLMclass".equals(values[valuesIdx])) {
                ++valuesIdx;
                continue;
            }
            if (fieldsIdx == 3) {
                topFields[fieldsIdx++] = "%Wait";
                continue;
            }
            topFields[fieldsIdx++] = DataHelper.newString(values[valuesIdx++]);
        }
        return topFields;
    }

    private void parseTopData(String[] values) {
        double[] recordData = new double[this.topFields.length];
        int n = 1;
        int pid = -1;
        String name = values[this.topCommandIndex];
        try {
            pid = Integer.parseInt(values[n++]);
            ++n;
            for (int i = 0; i < recordData.length; ++i) {
                if (n == this.topCommandIndex) {
                    ++n;
                }
                if (i == 3) {
                    recordData[i] = recordData[0] - recordData[1] - recordData[2];
                    if (!(recordData[i] < 0.0)) continue;
                    recordData[0] = recordData[0] - recordData[i];
                    recordData[i] = 0.0;
                    continue;
                }
                recordData[i] = Double.parseDouble(values[n++]);
            }
        }
        catch (NumberFormatException nfe) {
            LOGGER.warn("{}: invalid numeric data '{}' at line {}, column {}", new Object[]{this.currentRecord.getTimestamp(), values[n], this.in.getLineNumber(), n - 1});
            return;
        }
        Process process = this.processes.get(pid);
        boolean newProcess = false;
        ProcessDataType processType = null;
        if (process != null) {
            if (process.getName().equals(name)) {
                processType = this.data.getType(process);
            } else {
                LOGGER.debug("process id {} reused; '{}' is now '{}'", new Object[]{pid, process.getName(), name});
                process.setEndTime(this.currentRecord.getTime());
                newProcess = true;
            }
        } else {
            newProcess = true;
        }
        if (newProcess) {
            process = new Process(pid, this.currentRecord.getTime(), name);
            this.processes.put(pid, process);
            processType = new ProcessDataType(process, this.topFields);
            this.data.addType(processType);
            this.data.addProcess(process);
        }
        process.setEndTime(this.currentRecord.getTime());
        if (this.scaleProcessesByCPU) {
            this.currentRecord.addData(processType, this.scaleProcessDataByCPUs(processType, recordData));
        } else {
            this.currentRecord.addData(processType, recordData);
        }
    }

    private void parseUARG(String[] values) {
        int cmdLineIdx = 4;
        if (this.isAIX) {
            cmdLineIdx = 8;
        }
        if (values.length < cmdLineIdx) {
            return;
        }
        String commandLine = values[cmdLineIdx];
        for (int i = cmdLineIdx + 1; i < values.length; ++i) {
            commandLine = commandLine + ',' + values[i];
        }
        commandLine = DataHelper.newString(commandLine);
        int pid = -1;
        try {
            pid = Integer.parseInt(values[2]);
        }
        catch (NumberFormatException nfe) {
            LOGGER.warn("invalid process id {} at line {}", (Object)values[2], (Object)this.in.getLineNumber());
            return;
        }
        Process process = this.processes.get(pid);
        if (process == null) {
            LOGGER.warn("misplaced UARG record at line {}, no process with pid {} not defined yet", (Object)this.in.getLineNumber(), (Object)pid);
            return;
        }
        if ("".equals(process.getCommandLine())) {
            process.setCommandLine(commandLine);
        } else if (!process.getCommandLine().equals(commandLine)) {
            LOGGER.debug("process id {} reused; command line for '{}' is now '{}'", new Object[]{pid, process.getName(), commandLine});
            process.setEndTime(this.currentRecord.getTime());
            ProcessDataType oldProcessType = this.data.getType(process);
            process = new Process(process.getId(), this.currentRecord.getTime(), process.getName());
            process.setCommandLine(commandLine);
            ProcessDataType processType = new ProcessDataType(process, this.topFields);
            this.data.addType(processType);
            this.data.addProcess(process);
            double[] data = this.currentRecord.getData(oldProcessType);
            this.currentRecord.removeData(oldProcessType);
            this.currentRecord.addData(processType, data);
        } else {
            LOGGER.warn("command line for process id {} redefined at line {}", (Object)pid, (Object)this.in.getLineNumber());
        }
    }

    private DataType buildDataType(String[] values) {
        if (values.length < 3) {
            if (!values[0].startsWith("DG")) {
                LOGGER.warn("invalid data type definition, no fields defined at line {} for data {}", (Object)this.in.getLineNumber(), (Object)Arrays.toString(values));
            }
            return null;
        }
        if ("ERROR".equals(values[0])) {
            LOGGER.warn("not creating ERROR data type at line {} for data {}", (Object)this.in.getLineNumber(), (Object)Arrays.toString(values));
            return null;
        }
        String id = DataHelper.newString(values[0]);
        String name = values[1];
        int idx = name.indexOf(this.data.getHostname());
        name = idx != -1 ? DataHelper.newString(name.substring(0, idx - 1)) : DataHelper.newString(name);
        List<Integer> toSkip = TYPE_SKIP_INDEXES.get(id);
        if (toSkip == null) {
            toSkip = Collections.emptyList();
        }
        String[] fieldNames = new String[values.length - 2 - toSkip.size()];
        int n = 0;
        for (int i = 2; i < values.length; ++i) {
            if (toSkip.contains(i)) continue;
            fieldNames[n++] = DataHelper.newString(values[i]);
        }
        for (DataTransform transform : this.transforms) {
            if (!transform.isValidFor(id, null)) continue;
            return transform.buildDataType(id, null, name, fieldNames);
        }
        return new DataType(id, name, fieldNames);
    }

    private double[] scaleProcessDataByCPUs(ProcessDataType processType, double[] values) {
        DataType cpuAll;
        double CPUs = this.fileCPUs;
        if (this.isAIX) {
            cpuAll = this.data.getType("PCPU_ALL");
            if (this.currentRecord.hasData(cpuAll)) {
                CPUs = this.currentRecord.getData(cpuAll, "Entitled Capacity");
            }
        } else {
            cpuAll = this.data.getType("CPU_ALL");
            if (this.currentRecord.hasData(cpuAll)) {
                CPUs = (int)this.currentRecord.getData(cpuAll, "CPUs");
            }
        }
        if (CPUs > 1.0) {
            for (String field : processType.getFields()) {
                if (!field.startsWith("%")) continue;
                int n = processType.getFieldIndex(field);
                values[n] = values[n] / CPUs;
            }
        }
        return values;
    }

    private void completeCurrentRecord() {
        for (DataPostProcessor processor : this.processors) {
            processor.postProcess(this.data, this.currentRecord);
        }
        this.data.addRecord(this.currentRecord);
        this.currentRecord = null;
    }

    static {
        HashMap<String, List<Integer>> tempIndexes = new HashMap<String, List<Integer>>();
        tempIndexes.put("RAWLPAR", Collections.unmodifiableList(Arrays.asList(2, 3)));
        tempIndexes.put("RAWCPUTOTAL", Collections.singletonList(4));
        TYPE_SKIP_INDEXES = Collections.unmodifiableMap(tempIndexes);
    }
}

